const express = require("express");
const jwt = require("jsonwebtoken");
const bcrypt = require("bcrypt");
const router = express.Router();
const nodemailer = require("nodemailer");

//model import
const { TutorModel } = require("../models/tutor.model");

//middleware import
const { isTutorAuthenticated } = require("../middlewares/authenticate");

// ✅ Get all tutor data (supports subject filter)
router.get("/all", async (req, res) => {
  const { subject } = req.query; // changed from "filter" → "subject"
  try {
    let tutors;
    if (subject) {
      tutors = await TutorModel.find({ subject });
    } else {
      tutors = await TutorModel.find();
    }

    res.status(200).send({
      msg: "All tutor data fetched successfully",
      tutors,
    });
  } catch (error) {
    console.error("Error fetching tutors:", error);
    res.status(500).send({ msg: "Something went wrong while fetching tutors" });
  }
});

// ✅ Register new tutor (with mail + JWT protection)
router.post("/register", isTutorAuthenticated, async (req, res) => {
  const { name, email, password, subject } = req.body.data;

  try {
    // check if user already exists
    const existingTutor = await TutorModel.findOne({ email });
    if (existingTutor) {
      return res.send({ msg: "User already registered" });
    }

    // hash password
    const secure_password = await bcrypt.hash(
      password,
      +process.env.Salt_rounds
    );

    // save tutor
    const tutor = new TutorModel({
      name,
      email,
      subject,
      password: secure_password,
    });
    await tutor.save();

    // send mail asynchronously (without breaking response)
    const transporter = nodemailer.createTransport({
      host: "smtp.gmail.com",
      port: 587,
      secure: false,
      auth: {
        user: process.env.ADMIN_GMAIL,
        pass: process.env.ADMIN_PASSWORD,
      },
    });

    const mailOptions = {
      from: `${process.env.ADMIN_GMAIL}`,
      to: email,
      subject: "Account ID and Password",
      html: `<p>Welcome to LMS, your Tutor account has been created successfully.</p>
             <p><strong>Email:</strong> ${email}</p>
             <p><strong>Password:</strong> ${password}</p>`,
    };

    transporter.sendMail(mailOptions, (error, info) => {
      if (error) console.log("Mail Error:", error);
    });

    // ✅ Send single proper response
    res.status(201).send({
      msg: "Tutor registered successfully",
      tutor,
    });
  } catch (err) {
    console.error("Tutor Registration Error:", err);
    res.status(500).send({ msg: "Tutor Registration failed" });
  }
});

// ✅ Tutor login
router.post("/login", async (req, res) => {
  const { email, password } = req.body;
  try {
    const tutor = await TutorModel.findOne({ email });
    if (!tutor) return res.status(400).send({ message: "Wrong credentials" });

    if (tutor.access === "false") {
      return res.send({ message: "Access Denied" });
    }

    const match = await bcrypt.compare(password, tutor.password);
    if (!match) return res.status(400).send({ message: "Wrong credentials" });

    const token = jwt.sign(
      { email, name: tutor.name },
      process.env.secret_key,
      { expiresIn: "7d" }
    );

    res.send({ message: "Login Successful", user: tutor, token });
  } catch (error) {
    console.error(error);
    res.status(500).send({ message: "Server Error" });
  }
});

// ✅ Edit tutor
router.patch("/:tutorId", isTutorAuthenticated, async (req, res) => {
  const { tutorId } = req.params;
  const payload = req.body.data;
  try {
    await TutorModel.findByIdAndUpdate(tutorId, payload);
    const updatedTutor = await TutorModel.findById(tutorId);
    res.status(200).send({ msg: "Updated Tutor", tutor: updatedTutor });
  } catch (error) {
    res.status(500).send({ msg: "Error updating tutor" });
  }
});

// ✅ Delete tutor
router.delete("/:tutorId", async (req, res) => {
  const { tutorId } = req.params;
  try {
    await TutorModel.findByIdAndDelete(tutorId);
    res.status(200).send({ msg: "Deleted Tutor" });
  } catch (error) {
    res.status(500).send({ msg: "Error deleting tutor" });
  }
});

module.exports = router;
